<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;

class AuthController extends Controller
{

    // Регистрация
    public function register(Request $request)
    {
        $data = $request->all();

        $validator = Validator::make($data, [
            'first_name' => 'required|string|max:100',
            'last_name' => 'required|string|max:100',
            'patronymic' => 'nullable|string|max:100',
            'email' => 'required|string|email|max:150|unique:users',
            'password_hash' => 'required|string|min:8',
            'birth_date' => 'required|date',
        ], [
            'first_name.required' => 'Поле Имя обязательно для заполнения.',
            'last_name.required' => 'Поле Фамилия обязательно для заполнения.',
            'email.required' => 'Поле Email обязательно для заполнения.',
            'email.unique' => 'Этот Email уже занят.',
            'password.required' => 'Поле Пароль обязательно для заполнения.',
            'birth_date.required' => 'Поле Дата рождения обязательно для заполнения.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => [
                    'code' => 422,
                    'message' => 'Ошибка валидации',
                    'errors' => $validator->errors(),
                ],
            ], 422);
        }

        $user = new User();
        $user->first_name = $data['first_name'];
        $user->last_name = $data['last_name'];
        $user->patronymic = $data['patronymic'] ?? null;
        $user->birth_date = $data['birth_date'];
        $user->email = $data['email'];
        $user->password_hash = Hash::make($data['password_hash']);
        $user->role_id = 2;
        $user->access_token = Str::random(60);
        $user->status = true;

        if ($user->save()) {
            $user->setLogs('Аутентификация', 'Success!');
            return response()->json([
                'data' => [
                    'user' => [
                        'name' => "{$user->last_name} {$user->first_name} {$user->patronymic}",
                        'email' => $user->email,
                    ],
                    'code' => 201,
                    'message' => 'Пользователь создан',
                ],
            ], 201);
        }

        return response()->json([
            'error' => [
                'code' => 500,
                'message' => 'Внутренняя ошибка сервера',
            ],
        ], 500);
    }

    // Авторизация
    public function login(Request $request)
    {
        $data = $request->all();

        $validator = Validator::make($data, [
            'email' => 'required|string|email',
            'password' => 'required|string',
        ], [
            'email.required' => 'Поле Email обязательно для заполнения.',
            'password.required' => 'Поле Пароль обязательно для заполнения.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => [
                    'code' => 422,
                    'message' => 'Ошибка валидации',
                    'errors' => $validator->errors(),
                ],
            ], 422);
        }

        $user = User::where('email', $data['email'])->first();

        if (!$user || !Hash::check($data['password'], $user->password_hash)) {
            return response()->json([
                'error' => [
                    'code' => 401,
                    'message' => 'Неправильный логин или пароль',
                ],
            ], 401);
        }

        $user->access_token = Str::random(60);
        $user->save();

        $user->setLogs('Аутентификация', 'Пользователь успешно вошёл в аккаунт!');

        return response()->json([
            'data' => [
                'user' => [
                    'id' => $user->id,
                    'name' => "{$user->first_name} {$user->patronymic} {$user->last_name}",
                    'birth_date' => $user->birth_date,
                    'email' => $user->email,
                ],
                'token' => $user->access_token,
            ],
        ], 200);
    }

    // Logout
    public function logout(Request $request)
    {
        $user = Auth::guard('api')->user();

        if ($user) {
            $user->access_token = null;
            $user->save();

            $user->setLogs('Аутентификация', 'Выход из системы');

            return response()->noContent();
        }

        return response()->json([
            'message' => 'Ошибка авторизации',
        ], 403);
    }
}

