<?php

namespace App\Http\Controllers;

use App\Models\Session;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SessionController extends Controller
{
    public function create(Request $request)
    {

        // Валидация данных
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'type_id' => 'required|exists:session_types,id',
            'description' => 'required|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'max_participants' => 'required|integer|min:1',
        ], [
            'title.required' => 'Поле Название обязательно для заполнения.',
            'type_id.required' => 'Поле Тип обязательно для заполнения.',
            'description.required' => 'Поле Описание обязательно для заполнения.',
            'start_date.required' => 'Поле Дата начала обязательно для заполнения.',
            'end_date.required' => 'Поле Дата окончания обязательно для заполнения.',
            'max_participants.required' => 'Поле Максимальное количество участников обязательно для заполнения.',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => [
                    'code' => 422,
                    'message' => 'Ошибка валидации',
                    'errors' => $validator->errors(),
                ]
            ], 422);
        }

        // Создаем новый объект модели Session
        $session = new Session();
        $session->title = $request->input('title');
        $session->type_id = $request->input('type_id');
        $session->description = $request->input('description');
        $session->start_date = $request->input('start_date');
        $session->end_date = $request->input('end_date');
        $session->max_participants = $request->input('max_participants');

        // Сохраняем модель в базе данных
        if ($session->save()) {
            // Логирование (если необходимо)
            $session->setLogs('Создание', 'Занятие успешно создано!');

            return response()->json([
                'message' => 'Занятие успешно создано!',
                'data' => [
                    'id' => $session->id,
                    'title' => $session->title,
                    'type' => [
                        'name' => $session->sessionType->name
                    ],
                    'description' => $session->description,
                    'start_date' => $session->start_date,
                    'end_date' => $session->end_date,
                    'max_participants' => $session->max_participants,
                ]
            ], 201);
        } else {
            return response()->json([
                'error' => [
                    'code' => 500,
                    'message' => 'Произошла ошибка при сохранении данных.'
                ]
            ], 500);
        }
    }

    public function delete($id)
    {
        $session = Session::find($id);

        // Если занятие не найдено
        if (!$session) {
            return response()->json([
                'error' => [
                    'code' => 404,
                    'message' => 'Занятие не найдено',
                ]
            ], 404);
        }

        // Удаляем занятие
        if ($session->delete()) {
            // Логирование
            $session->setLogs('Удаление', 'Занятие успешно удалено!');

            return response()->json([
                'data' => [
                    'message' => 'Занятие успешно удалено',
                ]
            ], 204);
        }

        // Если не удалось удалить, возвращаем ошибку
        return response()->json([
            'error' => [
                'code' => 500,
                'message' => 'Ошибка при удалении занятия',
            ]
        ], 500);
    }

    public function update(Request $request, $id)
    {
        // Находим занятие по ID
        $session = Session::find($id);

        // Если занятие не найдено
        if (!$session) {
            return response()->json([
                'error' => [
                    'code' => 404,
                    'message' => 'Занятие не найдено',
                ]
            ], 404);
        }

        // Валидация данных
        $validator = Validator::make($request->all(), [
            'title' => 'required|string|max:255',
            'type_id' => 'required|exists:session_types,id',
            'description' => 'required|string',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'max_participants' => 'required|integer|min:1',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'error' => [
                    'code' => 422,
                    'message' => 'Ошибка валидации',
                    'errors' => $validator->errors(),
                ]
            ], 422);
        }

        // Обновляем данные модели
        $session->title = $request->input('title');
        $session->type_id = $request->input('type_id');
        $session->description = $request->input('description');
        $session->start_date = $request->input('start_date');
        $session->end_date = $request->input('end_date');
        $session->max_participants = $request->input('max_participants');

        // Сохраняем модель в базе данных
        if ($session->save()) {
            // Логирование (если необходимо)
            $session->setLogs('Обновление', 'Занятие успешно обновлено!');

            return response()->json([
                'message' => 'Занятие успешно обновлено',
                'data' => [
                    'id' => $session->id,
                    'title' => $session->title,
                    'type' => [
                        'name' => $session->sessionType->name
                    ],
                    'description' => $session->description,
                    'start_date' => $session->start_date,
                    'end_date' => $session->end_date,
                    'max_participants' => $session->max_participants,
                ]
            ], 200);
        } else {
            return response()->json([
                'error' => [
                    'code' => 500,
                    'message' => 'Ошибка при обновлении занятия',
                ]
            ], 500);
        }
    }

    public function search(Request $request)
    {
        // Получаем параметр query из GET-запроса
        $query = $request->query('query');

        // Если query не передан, возвращаем пустой результат
        if (!$query) {
            return response()->json([
                'data' => [],
            ]);
        }

        // Ищем курсы по названию и описанию
        $sessions = Session::where('title', 'like', "%{$query}%")
            ->orWhere('description', 'like', "%{$query}%")
            ->get();

        // Формируем результат
        $data = $sessions->map(function ($session) {
            return [
                'id' => $session->id,
                'title' => $session->title,
                'description' => $session->description,
                'type_id' => [
                    'name' => $session->sessionType->name
                ],
                'start_date' => $session->start_date,
                'end_date' => $session->end_date,
                'max_participants' => $session->max_participants,
            ];
        });

        return response()->json([
            'messages' => 'Данные успешно получены',
            'data' => $data,
        ]);
    }
}
